#
# Template.pm
#
# Copyright (c) 2001 by Michael Peppler. All rights reserved.
#
# $Id: //depot/http/lib/My/Template.pm#3 $
# $DateTime: 2001/06/25 11:26:36 $
# $Author: mpeppler $
#

package My::Template;

use strict;

use Apache::Constants qw(:common);
use Apache::Request;

use URI::Escape;

use Data::Dumper;

use My::Config;

use vars qw(@ISA);

#@ISA = qw(CGI My::Cookie); # Request Errors Cookie);


sub new {
    my $package     = shift;
    my $r           = shift;
    my %opt         = @_;

#    my $configFile  = $opt{ConfigFile} || $ENV{CONFIG_FILE};
    
    my $self = {};
    $self->{__APR} = Apache::Request->new($r);
    $self->{__R}    = $r;

    bless($self, $package);

    $self;
}

sub DESTROY {
    # No-op
}


# Make sure a header is printed only once
sub header {
    my $self = shift;
    my $type = shift;

    $type = 'text/html' unless $type;

    if(!$self->{__HEADER}) {
	$self->{__APR}->content_type($type);
	$self->{__APR}->send_http_header;
	$self->{__HEADER} = 1;
    }
}

sub redirect {
    my $self = shift;
    my $url  = shift;
    
    if($url !~ /^http:/) {
	my $server = $self->{__R}->server->server_hostname;
	$url = "http://$server$url";
    }

    $self->{__R}->send_cgi_header(<<EOT);
Location: $url

EOT
}

sub print {
    my $self = shift;
    my $text = shift;

    $self->{__APR}->print($text);
}

sub log_error {
    my $self = shift;
    $self->{__R}->log_error(@_);
}

sub warn {
    my $self = shift;
    $self->{__R}->warn(@_);
}

sub set {
    my $self = shift;
    my $key  = shift;
    my @val = @_;

    if(@val == 1) {
	$self->{__VALUES}->{$key} = $val[0];
    } else {
	$self->{__VALUES}->{$key} = [@val];
    }
}

sub unset {
    my $self = shift;
    my $key  = shift;

    delete($self->{__VALUES}->{$key});
    $self->{__APR}->param($key => undef);
}

sub push {
    my $self = shift;
    my $key  = shift;
    my @val = @_;

    if($self->{__VALUES}->{$key} && 
       ref($self->{__VALUES}->{$key}) ne 'ARRAY' ) 
    {
	my $t = $self->{__VALUES}->{$key};
	$self->{__VALUES}->{$key} = [$t];
    } 
    push(@{$self->{__VALUES}->{$key}}, @val);
}

sub get {
    my $self = shift;
    my $key  = shift;
    my $pos  = shift;
    
    my ($val, @d);
    if(exists($self->{__VALUES}->{$key})) {
	$val = $self->{__VALUES}->{$key};
    } elsif((@d = $self->{__APR}->param($key)) ){
	if(@d > 1) {
	    $val = [@d];
	} else {
	    $val = $d[0];
	}
    } elsif(my $cxt = $self->{__CONTEXT}) {
	$val = $cxt->get($key);
    }

    if(wantarray) {
	if(ref($val) eq 'ARRAY') {
	    return @$val;
	} else {
	    return ($val);
	}
    }

    if(ref($val) eq 'ARRAY') {
	$pos = 0 unless $pos;
	$val = $$val[$pos];
    }

    $val;
}

sub initTable {
    my $self  = shift;
    my $table = shift;
    my $rows  = shift;

    $self->{__TABLE}->{$table} = $rows;
}

sub fatal {
    my $self = shift;
    my $string = shift;
    my $req = shift;

    $self->set('error_string', $string);

    my $email = My::Config->get('fatal_email');

    open(MAIL, "|/usr/sbin/sendmail -odq -t") || die "Can't run sendmail: $!";
    print MAIL "To: $email\n";
    print MAIL "Subject: Fatal Error\n\n";
    print MAIL "Environment:\n";
    foreach (sort(keys(%ENV))) {
	print MAIL "\t$_:\t$ENV{$_}\n";
    }
    print MAIL "\nQuery object:\n";
    print MAIL Dumper($self);
    if($req) {
	print MAIL "\nRequest object:\n";
	print MAIL Dumper($req);
    }
    print MAIL "\nStack trace:\n";
    print MAIL Carp::longmess();
    close(MAIL);
    my $html = $self->parseTemplate("$self->{__ROOT}/fatal.html");
    $self->header;
    $self->print($html);
}

sub errorHandler {
    my $self = shift;
    my $req  = shift;

    if($req && UNIVERSAL::isa($req, 'My::Request')) {
	my $status = $req->getStatus;
	return unless $status;

#	$self->set('error_string', $req->getErrorString());

	if($status > 100 || $status < 0) {
	    $self->fatal($req->getErrorString, $req);
	    $self->{__ERROR} = 1;
	    die "Fatal error: ", $req->getErrorString();
	}
    }
}

sub parseTemplate {
    my $self         = shift;
    my $templateFile = shift;
    my $hiddenVars   = shift;

    my $templatePage = '';
    my $ssi = 0;
    my $ssiPath;
    
    $self->addHiddenToTemplate($hiddenVars);

    local $_;

    if( open(TEMPLATE, $templateFile) ){
	while(<TEMPLATE>){
	    my $row = $_;

#	    warn "$row\n";
	    if($row =~ /<!--\#include\s+virtual=(\S+)\s*-->/i) {
		$row = $self->processInclude($1);
	    }

	    if($row =~ /<!--\s+MyTable:\s+start\s+(\w+)\s+(\d+)/) {
		my $proc = $1;
		my $rows = $2;
		my $table = '';
		while(<TEMPLATE>) {
		    last if(/<!--\s+MyTable:\s+end\s+$proc/);
		    $table .= $_;
		}
		$row .= $self->processTable($proc, $rows, $table);
	    }
	    if( $row =~ /<form.*>/i ) {
		$row = $self->addFormHidden($row);
	    }
	    $row = $self->replaceTemplateVar($row);
	    $templatePage .= $row;
	}
	close(TEMPLATE);	
    } else { 
	$self->header;
	print "Can't Open Template: $templateFile: $!<br>";
    }

    $templatePage;
}

sub processTable {
    my $self  = shift;
    my $proc  = shift;
    my $rows  = shift;
    my $table = shift;

    my $this_row;
    my $data = '';

    my $cur = 0;

    if(!defined($self->{__TABLE}->{$proc})) {
	print "Table $proc has not been defined.<br>", 
	return $table
    }
    if($rows > $self->{__TABLE}->{$proc}) {
	$rows = $self->{__TABLE}->{$proc};

    }

    while($cur < $rows) {
	$this_row = $table;
	$this_row = $self->replaceTemplateVar($this_row, $cur);

	$data .= $this_row;

	++$cur;
    }
    $data;
}

sub addHiddenToTemplate {
    my $self    = shift;
    my $hiddens = shift;

    my %seen;
    grep($seen{$_} = 1, @{$self->{__HIDDEN}});
    
    push(@{$self->{__HIDDEN}}, grep(!$seen{$_}, @$hiddens))
	if $hiddens && ref($hiddens) eq 'ARRAY';
}



sub addFormHidden {
    my $self= shift;
    my $row = shift;

    my %seen;
    
    foreach (@{$self->{__HIDDEN}}){
	next if(exists($seen{$_})); # only place each var once!
	$seen{$_} = 1;
	my $val = $self->get($_);
	$val = '' unless defined($val);	# silence -w
	$row .= qq(<INPUT NAME="$_" TYPE="HIDDEN" VALUE="$val">\n);
    }
    
    $row;
}


sub replaceTemplateVar {    
    my $self  = shift;
    my $html  = shift;
    my $row   = shift;
    
    while( $html =~ /({my::\w+?::my})/ ) {
	my $templateVar = $1;
	my ($funky) = ($templateVar =~ /{my::(\w+?)::my}/);

	my $val = $self->get($funky, $row);
	$val = '' unless defined $val;	# shut up -w
	$html =~ s/$templateVar/$val/;
    }
    
    $html;    
}

sub processInclude {
    my $self = shift;
    my $file = shift;

#    warn "processInclude ($file)";

    $file =~ s/\"//g;

    if($file =~ /^\//) {
	$file = "$self->{__ROOT}$file";
    }

    open(IN, $file) || do {
	warn "processInclude: can't open $file\n";
	return '';
    };
    local $/ = undef;
    my $data = <IN>;
    close(IN);

    return $data;
}

1;
