#
# $Id$
#
package My::Request;

use strict;

use Data::Dumper;

use My::Config;
use My::Error;
use Sybase::CTlib;

use Carp;

use vars qw($AUTOLOAD);

my $verbose = My::Config->get('DbVerbose') || 0;
my $errorStr;

ct_callback(CS_CLIENTMSG_CB, \&msg_cb);
ct_callback(CS_SERVERMSG_CB, \&srv_cb);

my $fileDir = My::Config->get('serviceDir');
my $status;

require "$fileDir/request.ix";

AUTOLOAD {
    my $sub = $AUTOLOAD;
    my $filename;

    {
	my ($pkg,$func) = ($sub =~ /(.*)::([^:]+)$/);
	$filename = "$fileDir/$func.al";
    }

    my $save = $@;
    eval { local $SIG{__DIE__}; require $filename };
    if ($@) {
	if (substr($sub,-9) eq '::DESTROY') {
	    no strict 'refs';
	    *$sub = sub {};
	} else {
	    if ($@){
		$@ =~ s/ at .*\n//;
		my $error = $@;
		require Carp;
		Carp::croak($error);
	    }
	}
    }
    $@ = $save;
    goto &$sub;
}

sub new {
    my $package = shift;
    my $query   = shift;

    bless({__QUERY => $query}, $package);
}

sub getStatus {
    my $self = shift;

    return $self->{STATUS}->{APIRETCODE};
}

sub getErrorString {
    my $self = shift;

    return $self->{STATUS}->{error_text};
}

sub next {
    my $self = shift;
    my $resname = shift;

    if($self->{DATA}->{$resname} && ref($self->{DATA}->{$resname})) {
	return shift @{$self->{DATA}->{$resname}};
    }

    return undef;
}

sub array_ref {
    my $self = shift;
    my $resname = shift;

    if($self->{DATA}->{$resname} && ref($self->{DATA}->{$resname})) {
	return $self->{DATA}->{$resname};
    }

    return undef;
}

sub execute {
    my $self   = shift;
    my $config = shift;
    my $args = shift;

    my @list;
    my @missing;

#    warn Dumper($args);

    my $param = $config->{param};

    foreach my $k (keys(%$param)) {
	my $val = $args->{$k};
	if(!defined($val) ||
	   ($val eq '' && $param->{$k}->{type} != CS_CHAR_TYPE))
	{
	    if(!$param->{$k}->{opt}) {
		push(@missing, $k);
	    }
	    next;
	}
	if($param->{$k}->{type} == CS_BINARY_TYPE && $val =~ /^0x/i) {
	    $val = pack('H*', substr($val, 2));
	}
	push(@list, { name => '@' . $k,
		      status => CS_INPUTVALUE,
		      value => $val,
		      datatype => $param->{$k}->{type},
		      indicator => CS_UNUSED });
    }
    if(@missing) {
	confess("Request $config->{name} is missing some parameters: @missing\n");
    }

    my $db  = My::Config->get("database_$config->{database}");
    my $srv = My::Config->get("server_$config->{server}");

    # Clear existing data from the $req object (if any)
    $self->{STATUS} = {};	# XXX
    $self->{DATA}   = {};	# XXX
    $self->procCall($srv, $db, $config->{name}, \@list,
		    $config->{output});

    if($self->getStatus() && $self->{__QUERY}) {
	if($self->{__QUERY}->can('errorHandler')) {
	    $self->{__QUERY}->errorHandler($self);
	}
    }
}


sub procCall {
    my $self = shift;
    my $srv  = shift;
    my $db   = shift;
    my $proc = shift;
    my $param  = shift;
    my $output = shift;

    $status = {};

    my $handle = My::Request::DB->connect($srv);

    if(!defined($handle) || !defined($handle->{DBH})) {
	$status->{error_text} = "[DB] error: can't allocate a database handle\n";
	$status->{error_code} = My::Error->getCode('SYS_DB_NO_HANDLE'); ## XXX

	$self->{STATUS} = $status;
	
	return;
    }
    my $dbh = $handle->{DBH};

    if($verbose) {
	print STDERR "[DB] exec $db..$proc\n";
	foreach my $p (@$param) {
	    print STDERR "[DB]\t$p->{name} = $p->{value}\n";
	}
    }

    if($dbh->ct_command(CS_RPC_CMD, "$db..$proc", CS_NULLTERM, CS_NO_RECOMPILE) != CS_SUCCEED) {
	$status->{error_text} = "[DB] error: $errorStr\n";
	$status->{error_code} = 200;  #XXX;

	$self->{STATUS} = $status;

	return;
    }

    foreach my $p (@$param) {
	if($dbh->ct_param($p) != CS_SUCCEED) {
	    $status->{error_text} = "[DB] error: $errorStr\n";
	    $status->{error_code} = 200;  #XXX;

	    $self->{STATUS} = $status;

	    return;
	}
    }


    if($dbh->ct_send() != CS_SUCCEED) {
	$status->{error_text} = "[DB] error: $errorStr\n";
	$status->{error_code} = 200;  #XXX;

#	disconnect($dbh);

	$self->{STATUS} = $status;
	
#	return {STATUS => $status};
	return;
    }

    my $restype;
    my $retstatus;
    my $resnum = 0;
    my $call;

    while(($call = $dbh->ct_results($restype)) == CS_SUCCEED) {

	print STDERR "[DB] ct_results($restype)\n" if($verbose > 1);

	if($restype == CS_CMD_FAIL) {
	    $dbh->ct_cancel(CS_CANCEL_ALL);
	    $status->{error_text} = "[DB] error: $errorStr\n";
	    $status->{error_code} = 101;  #XXX;

	    next;
	}

	next unless($dbh->ct_fetchable($restype));

	if($restype == CS_STATUS_RESULT) {
	    while(my $d = $dbh->ct_fetch(0,1)) {
		$retstatus = $d->[0];
	    }
	    print STDERR "[DB] STATUS: $retstatus\n" if $verbose;
	    $status->{PROC_STATUS} = $retstatus;
	    $status->{APIRETCODE} = $retstatus;
	    if(!ref($output->[0]->{cols})) {
		push(@{$self->{DATA}->{$output->[0]->{name}}}, $status);
	    }
	} else {
	    local $^W = 0;
	    my %data;
	    my $col;
	    my $found = 0;

	    if($verbose) {
		my @names = $dbh->ct_col_names();
		print STDERR "[DB] COLS: @names\n";
	    }
	    while(my $d = $dbh->ct_fetch(0, 1)) {
		++$found;
		print STDERR "[DB] DATA: @$d\n" if $verbose;
		next unless ref($output->[$resnum]->{cols});
		%data = ();
		for(my $i = 0; $i < @$d; ++$i) {
		    $col = $output->[$resnum]->{cols}->[$i];
		    next unless $col;
		    $data{$col} = $d->[$i];
		}
	
		push(@{$self->{DATA}->{$output->[$resnum]->{name}}}, {%data});
	    }

	    ++$resnum;
	}
    }

    my $tran = $dbh->ct_sql('select @@transtate, @@trancount');
    if(defined($tran) && defined($tran->[0]->[0]) && $tran->[0]->[0] == 0) {
	my $error = "A transaction is still pending, with \@\@trancount = $tran->[0]->[1] for proc $proc - rolling back";
      Carp::cluck("[DB] $error");
	$dbh->ct_sql('rollback tran');

	# force a TAP...
	$status->{error_text} = "[DB] error: $error\n";
	$status->{error_code} = My::Error->getCode('SYS_DB_BAD_TRANSACTION_STATE');  #XXX;
    }

    if($call == CS_FAIL) {
	$status->{error_text} = "[DB] error: $errorStr\n";
	$status->{error_code} = 101;  #XXX;
    }	

#    disconnect($dbh);

    $self->{STATUS} = $status;

}


# Utility callbacks to handle errors in OpenCLient and server errors.
sub msg_cb
{
    my($layer, $origin, $severity, $number, $msg, $osmsg, $dbh) = @_;

    $errorStr = sprintf("OC: %d %d %s", $number, $severity, $msg);
    if(defined($osmsg)) {
	$errorStr .= " OS: $osmsg";
    }

    warn "[DB] $errorStr\n";

    if($dbh && ref($dbh) && $number == 63) {
	warn "[DB] Got timeout - canceling request\n";
	$dbh->ct_cancel(CS_CANCEL_ATTN);
    }

    $status->{error_text} .= "$errorStr\n";
    # force a TAP...
    $status->{error_code} = My::Error->getCode('SYS_DB_TIMEOUT');

    CS_SUCCEED;
}

sub srv_cb
{
    my($dbh, $number, $severity, $state, $line, $server, $proc, $msg)
	= @_;

    if($severity > 10) {
	$errorStr = sprintf("%d %d %d %d %s %s %s",
			    $number, $severity, $state, $line,
			    $server, $proc, $msg);
	warn "[DB] $errorStr\n";

	$status->{error_text} .= "$errorStr\n";
    } elsif ($number == 0) {
	warn "[DB] $msg\n";
    }

    CS_SUCCEED;
}


1;



package My::Request::DB;

use Sybase::CTlib;

sub connect {
    my $package = shift;
    my $srv     = shift;

    my $dbh;
    my $is_conpool = 0;
    if($ENV{MOD_PERL} && $ENV{MOD_PERL} =~ /mod_perl/ &&
       defined(&Apache::Sybase::ConPool::getDbh)) 
    {
	warn "Fetching a connection from ConPool:\n";
	$dbh = Apache::Sybase::ConPool::getDbh($srv);
	warn "Got $dbh\n";
	$is_conpool = 1 if($dbh);
    }
    if(!$dbh) {
	warn "Opening a connection directly:\n";
	$dbh = new Sybase::CTlib 'webuser', 'webuser', $srv;
    }

    bless({DBH => $dbh, is_conpool => $is_conpool}, $package);
}

sub DESTROY {
    my $self = shift;

    if($self->{is_conpool}) {
	Apache::Sybase::ConPool::freeDbh($self->{DBH});
    } else {
	delete($self->{DBH});
    }
}

1;
