package My::BugDB;

use strict;

use Apache::Constants qw(:common);
use Apache::Request;
use Apache::Cookie;
use Digest::MD5 qw(md5_hex);

#use Apache::Sybase::ConPool;

use My::Request;
use My::Template;

use subs qw(index add view query login register);

my %commands = ( 'index' => \&index,
		 'add' => \&add,
		 'view' => \&view,
		 'query' => \&query,
#		 'edit' => \&edit,
		 'login' => \&login,
		 'register' => \&register);

my $secret = '55%321=)';

my $severity_list = [
	    { id => 10,
	      name => 'blocker', },
	    { id => 9,
	      name => 'critical', },
	    { id => 8,
	      name => 'major' },
	    { id => 7,
	      name => 'normal' },
	    { id => 6,
	      name => 'minor', },
	    { id => 5,
	      name => 'trivial', },
	    { id => 4,
	      name => 'enhancement' },
		];

my $priority_list = [
		 { id => 1,
		   name => 'P1', },
		 { id => 2,
		   name => 'P2', },
		 { id => 3,
		   name => 'P3' },
		 { id => 4,
		   name => 'P4', },
		 { id => 5,
		   name => 'P5', },
		     ];

my $status_list = [ 
	       { id => 0,
		 name => 'Unconfirmed', },
	       { id => 1,
		 name => 'New', },
	       { id => 2,
		 name => 'Assigned', },
	       { id => 3,
		 name => 'Re-opened', },
	       { id => 4,
		 name => 'Resolved', },
	       { id => 5,
		 name => 'Verified', },
	       { id => 6,
		 name => 'Closed', },
		  ];

my $resolution_list = [
#		   { id => 0,
#		     name => '', },
		   { id => 1,
		     name => 'Fixed', },
		   { id => 2,
		     name => 'Invalid', },
		   { id => 3,
		     name => "Won't fix", },
		   { id => 4,
		     name => 'Later', },
		   { id => 5,
		     name => 'Remind', },
		   { id => 6,
		     name => 'Duplicate', },
		   { id => 7,
		     name => 'Works for me', },
		      ];
		     
		 

sub handler {
    my $r = shift;

#    $r->register_cleanup(\&Apache::Sybase::ConPool::clearAll);

    my $query = My::Template->new($r);

    $query->{__ROOT} = My::Config->get('templateDir') . '/bugdb';

    checkCookie($query);

    my @params = $query->{__APR}->param;
    $r->warn("BugDB: $ENV{PATH_INFO}?@params");

    my $cmd = $ENV{PATH_INFO};
    $cmd =~ s|^/||;
    $r->warn("BugDB: cmd = $cmd");
    $cmd = 'index' unless $cmd;

    if(!$commands{$cmd}) {
	$r->log_error("BugDB: unknown action $cmd received");
	return NOT_FOUND;
    }

    eval {
	&{$commands{$cmd}}($query);
    };
    if($@) {
	my $err = $@;
	if(!$query->{__ERROR}) {
	    $query->fatal($err);
	}
	$r->log_error("[BugDB][$cmd]: $err");
	return OK;
    }

    return OK;
}

sub index {
    my $query = shift;

    my $html = $query->parseTemplate("$query->{__ROOT}/main.html");
    $query->header;
    $query->print($html);
}

sub add {
    my $query = shift;

    my $state = $query->get('s');
    my $err;

    if(!$query->{__UID}) {
	$query->redirect("/bugdb/login");
    }

 LOOP: {
	if(!$state) {
	    loadBugPD($query);
	    $query->set('s', 1);
	    my $html = $query->parseTemplate("$query->{__ROOT}/addbug.html",
					     [qw(s)]);
	    $query->header;
	    $query->print($html);
	} elsif($state == 1) {

#	    warn Data::Dumper::Dumper($query);
	    my $req = new My::Request $query;

	    $req->addBug(user_id => $query->{__UID},
			 os_id => $query->get('bug_os'),
			 component_id => $query->get('bug_component'),
			 platform_id => $query->get('bug_platform'),
			 product_id => $query->get('bug_product'),
			 summary => $query->get('bug_summary'),
			 description => $query->get('bug_description'),
			 severity => $query->get('bug_severity'),
			 priority => $query->get('bug_priority'),
			 version => $query->get('bug_version'));

	    if($req->getStatus) {
#		$query->set('error_string', $req->getErrorString());
		$state = 0;
		redo LOOP;
	    }

	    my ($bug) = $req->next('addBug');

	    $query->set('bug_id', $bug->{bug_id});
	    $query->set('email', $query->{__EMAIL});
	    my $html = $query->parseTemplate("$query->{__ROOT}/addbugdone.html");
	    $query->header;
	    $query->print($html);

	    sendEmail($query, $bug->{bug_id}, 'new');
	}
    }
}

sub login {
    my $query = shift;

    my $state = $query->get('s');

 LOOP: {
	if(!$state) {
	    my $html = $query->parseTemplate("$query->{__ROOT}/login.html");
	    $query->header;
	    $query->print($html);
	} elsif($state == 1) {
	    # perform login!
	    my $email = $query->get('email');
	    my $pwd = $query->get('pwd');
	    
	    my $req = My::Request->new($query);
	    $req->login(email => $email, password => $pwd);
	    if($req->getStatus) {
		$query->log_error("[BugDB] login failed");
		$state = 0;
		redo LOOP;
	    }
	    my ($data) = $req->next('login');
	    # set cookie, return to main page
	    setCookie($query, $email, $data->{user_id}, $data->{priviledge});
	    $query->redirect("/bugdb/");
	}
    }
}

sub register {
    my $query = shift;

    my $state = $query->get('s');

 LOOP: {
	if(!$state) {
	    my $html = $query->parseTemplate("$query->{__ROOT}/register.html");
	    $query->header;
	    $query->print($html);
	} elsif($state == 1) {
	    my $email = $query->get('email');
	    my $pwd = $query->get('pwd');
	    my $pwd2 = $query->get('pwd2');
	    
	    # First - verify that the two password values match:
	    if($pwd ne $pwd2) {
		$query->set('error_string', "The two password values don't match!");
		$state = 0;
		redo LOOP;
	    } else {
		my $req = My::Request->new($query);
		$req->addUser(email => $email, password => $pwd);
		if($req->getStatus) {
		    $query->log_error("[BugDB] addUser failed");
		    $state = 0;
		    redo LOOP;
		}
	    }
	    # set cookie, return to main page
	    setCookie($query, $email);
	    $query->redirect("/bugdb/");
	    return;
	}
    }
}

sub view {
    my $query = shift;

    my $state = $query->get('s');
    my $err;

 LOOP: {
	if(!$state) {
	    my $req = new My::Request $query;

	    $req->viewBug(bug_id => $query->get('bug_id'));

	    my $bug = $req->next('bug');
	    foreach (keys(%$bug)) {
		$query->set($_, $bug->{$_});
	    }
	    $query->set('bug_product', $bug->{product_id});
	    $query->set('bug_component', $bug->{component_id});
	    $query->set('bug_platform', $bug->{platform_id});
	    $query->set('bug_os', $bug->{os_id});
	    $query->set('bug_severity', $bug->{severity});
	    $query->set('bug_priority', $bug->{priority});
	    $query->set('bug_status', $bug->{status});
	    $query->set('bug_resolution', $bug->{resolution});
	    $query->set('bug_version', $bug->{version});
	    $query->set('bug_summary', $bug->{summary});
	    html_escape(\$bug->{description});
	    $query->set('bug_description', $bug->{description});

	    my $bug_log = $req->array_ref("bug_log");
	    if($bug_log && ref($bug_log) eq 'ARRAY') {
		$query->initTable("LogEntries", scalar(@$bug_log));
		foreach (@$bug_log) {
		    $query->push('log_date', $_->{log_date});
		    $query->push('log_email', $_->{email});
		    $query->push('log_text', $_->{log_text});
		}
	    }
	    
	    loadBugPD($query);

	    $query->set('status_pd',  
			buildPopDown($status_list, 'id', 'name', 
				     $query->get('bug_status')));
	    $query->set('resolution_pd',  
			buildPopDown($resolution_list, 'id', 'name', 
				     $query->get('bug_resolution')));

	    my $user_req = new My::Request $query;
	    $user_req->listUsers();
	    my $user_list = $user_req->array_ref("users");
	    $query->set('user_list_pd',
			buildPopDown($user_list, 'user_id', 'email',
				     $bug->{owner_id}));
	    my $cc_req = new My::Request $query;
	    $cc_req->listCc(bug_id => $query->get('bug_id'));
	    my $cc_list = $cc_req->array_ref('cc_list');
	    $query->set('cc_list_pd',
			buildPopDown($cc_list, 'email', 'email', undef));

	    $query->set('bug_status', id2val($status_list, $bug->{status}));
	    $query->set('bug_resolution', id2val($resolution_list, $bug->{resolution}));

	    if($bug->{status} <= 1) {
		$query->set("action_block", 
			    $query->parseTemplate("$query->{__ROOT}/bugact_new.html"));
	    } elsif($bug->{status} <= 3) {
		$query->set('action_block',
			    $query->parseTemplate("$query->{__ROOT}/bugact_assigned.html"));
	    } elsif($bug->{status} == 4) {
		$query->set('action_block',
			    $query->parseTemplate("$query->{__ROOT}/bugact_resolved.html"));
	    } elsif($bug->{status} == 5) {
		$query->set('action_block',
			    $query->parseTemplate("$query->{__ROOT}/bugact_verified.html"));
	    } elsif($bug->{status} == 6) {
		$query->set('action_block',
			    $query->parseTemplate("$query->{__ROOT}/bugact_closed.html"));
	    }

	    $query->set('s', 1);
	    my $html = $query->parseTemplate("$query->{__ROOT}/viewbug.html",
					     [qw(s timestamp bug_id)]);
	    $query->header;
	    $query->print($html);
	} elsif($state == 1) {
	    if(!$query->{__UID}) {
		$query->redirect("/bugdb/login");
	    }
	    if($query->{__PRIVS} > 0) {
		my $req = new My::Request $query;
		my $resolution;
		my $status;
		my $dup_of;
		my $owner_id;

		my $bug_id = $query->get('bug_id');

		warn "bug_id => $bug_id";

		my $new_cc = $query->get('new_cc');
		if($new_cc && $new_cc =~ /\S+\@\w+/) {
		    $req->addCc(bug_id => $bug_id,
				email => $new_cc);
		}
		if($query->get('del_cc')) {
		    my @cc_list = $query->get('cc');
		    foreach my $cc (@cc_list) {
			$req->delCc(bug_id => $bug_id,
				    email => $cc);
		    }
		}
			

		my $action = $query->get('action');
		if($action eq 'accept') {
		    $status = 2; # accepted
		} elsif($action eq 'resolve') {
		    $status = 4; # resolved
		    $resolution = $query->get('bug_resolution');
		} elsif($action eq 'dup') {
		    $status = 4;
		    $resolution = 6; # duplicate
		    $dup_of = $query->get('bug_duplicate');
		    if(!$dup_of || $dup_of =~ /\D/) {
			$query->set('error_string', "Invalid value '$dup_of'<br>Please enter the bug ID that this bug is a duplicate of");
			$state = 0;
			redo LOOP;
		    }
		} elsif($action eq 'reassign') {
#		    $status = 1; # actually should be left as "reopened" if
#				 # current status is reopened...
		    $owner_id = $query->get('bug_assignee');
		} elsif($action eq 'close') { 
		    $status = 6;
		} elsif($action eq 'reopen') { 
		    $status = 3;
		    $resolution = 0;
		} else {
		    # no-op - leave as is...
		}

		$req->updateBug(bug_id => $bug_id,
				user_id => $query->{__UID},
				owner_id => $owner_id,
				os_id => $query->get('bug_os'),
				component_id => $query->get('bug_component'),
				platform_id => $query->get('bug_platform'),
				product_id => $query->get('bug_product'),
				severity => $query->get('bug_severity'),
				status => $status,
				priority => $query->get('bug_priority'),
				resolution => $resolution,
				duplicate_of => $dup_of,
				log_text => $query->get('bug_log'),
				version => $query->get('bug_version'),
				timestamp => $query->get('timestamp'));
		if($req->getStatus() == 0) {
		    sendEmail($query, $bug_id, 'change');
		}
	    } else {
		my $log = $query->get('bug_log');
		if($log =~ /\w/) {	# not empty
		    my $req = new My::Request $query;
		    my $bug_id = $query->get('bug_id');
		    $req->addBugLog(bug_id => $bug_id,
				    user_id => $query->{__UID},
				    log_text => $query->get('bug_log'));
		    if($req->getStatus() == 0) {
			sendEmail($query, $bug_id, 'logentry');
		    }
		} else {
		    $query->set('error_string', "Please enter a log message before pressing <i>submit</i>");
		}
	    }
	    $state = 0;
	    redo LOOP;
	}
    }
}

sub query {
    my $query = shift;

    my $req = new My::Request $query;
    $req->listBugs;
    
    my $bug_list = $req->array_ref("listBugs");
    if($bug_list && ref($bug_list) eq 'ARRAY') {
	$query->initTable("buglist", scalar(@$bug_list));
	foreach (@$bug_list) {
	    $query->push('bug_id', $_->{bug_id});
	    $query->push('create_date', $_->{create_date});
#	    $query->push('owner', $_->{owner});
	    $query->push('product', $_->{product});
	    $query->push('priority', id2val($priority_list, $_->{priority}));
	    $query->push('status', id2val($status_list, $_->{status}));
	    $query->push('summary', $_->{summary});
	}
    }
    my $html = $query->parseTemplate("$query->{__ROOT}/listbugs.html");

    $query->header;
    $query->print($html);
}


# --- Utility Functions ---

sub setCookie {
    my $query  = shift;
    my $email  = shift;
    my $userId = shift;
    my $privs  = shift;

    my $hash = md5_hex(join(':', $email, $userId, $privs, $secret));
    my $r = $query->{__R};
    $r->warn("setCookie: $email, $userId, $privs, $hash");
    my $cookie = Apache::Cookie->new($r,
				     -name => 'bugdb',
				     -path => '/bugdb/',
				     -value => ['email', $email,
						'uid', $userId,
						'pr', $privs,
						'hash', $hash]
				     );

    $cookie->bake;
}

sub checkCookie {
    my $query = shift;
    my %cookies = Apache::Cookie->fetch;

    if(%cookies && $cookies{bugdb}) {
	my %ticket = $cookies{bugdb}->value;
	if($ticket{hash} && $ticket{email} && $ticket{uid}) {
	    my $hash = md5_hex(join(':', $ticket{email}, $ticket{uid}, 
				    $ticket{pr}, $secret));
	    if($hash eq $ticket{hash}) {
		$query->{__EMAIL} = $ticket{email};
		$query->{__UID}   = $ticket{uid};
		$query->{__PRIVS} = $ticket{pr};
	    }
	}
    }

    $query->{__EMAIL};
}

sub loadBugPD {
    my $query = shift;

    my $req = new My::Request $query;

    $req->getDefs;

    if($req->getStatus) {
	$query->set('error_string', $req->{STATUS}->{error_text});
    } else {
	my @d;
	while(my $d = $req->next('product')) {
	    push(@d, $d);
	}
	$query->set('product_pd', buildPopDown(\@d, 'product_id', 'name',
					      $query->get('bug_product')));
	@d = ();
	while(my $d = $req->next('component')) {
	    push(@d, $d);
	}
	$query->set('component_pd', buildPopDown(\@d, 'component_id', 'name',
						$query->get('bug_component')));
	@d = ();
	while(my $d = $req->next('platform')) {
	    push(@d, $d);
	}
	$query->set('platform_pd', buildPopDown(\@d, 'platform_id', 'name',
					       $query->get('bug_platform')));
	@d = ();
	while(my $d = $req->next('os')) {
	    push(@d, $d);
	}
	$query->set('os_pd', buildPopDown(\@d, 'os_id', 'name', 
					  $query->get('bug_os')));
    }
    $query->set('severity_pd',  buildPopDown($severity_list, 'id', 'name', 
					  $query->get('bug_severity') || 7));
    $query->set('priority_pd',  buildPopDown($priority_list, 'id', 'name', 
					  $query->get('bug_priority') || 3));
}

sub buildPopDown {
    my $data = shift;
    my $key = shift;
    my $val = shift;
    my $current = shift;

    my $opt = '';
    my $selected;

    return '' unless ref($data) eq 'ARRAY';

    foreach my $d (@$data) {
	if(defined($current) && $d->{$key} eq $current) {
#	    warn "buildPullDown: $d->{$key} - $d->{$val} == $current\n";
	    $selected = 'selected';
	} else {
	    $selected = '';
	}
	$opt .= qq(<option value="$d->{$key}" $selected>$d->{$val}</option>\n);
    }

    $opt;
}

sub id2val {
    my $data = shift;
    my $id   = shift;

    foreach (@$data) {
	if($_->{id} == $id) {
	    return $_->{name};
	}
    }
}

sub html_escape {
    my $strref = shift;

    $$strref =~ s/</&lt;/g;
    $$strref =~ s/>/&gt;/g;
    $$strref =~ s/\n/<br>/g;
}

sub sendEmail {
    my $query  = shift;
    my $bug_id = shift;
    my $type   = shift;

    my $email;
    my $subject; # = "New or updated bug $bug_id";
    my $recipients;
    my @cc;

    my $req = new My::Request $query;

    if($type eq 'new') {
	$req->viewBug(bug_id => $bug_id);
	my ($bug) = $req->next('bug');
	$subject = "[bug $bug_id] New: $bug->{summary}";

	push(@cc, $bug->{owner});

	$email = <<EOMAIL;
New bug $bug_id entered by $bug->{creator}

    http://www.peppler.org/bugdb/view?bug_id=$bug_id

Platform: $bug->{platform}      Product: $bug->{product}
      OS: $bug->{os}          Component: $bug->{component}
 Summary: $bug->{summary}

$bug->{description}

EOMAIL
	
    } elsif($type eq 'change') {
	$req->viewBugDiff(bug_id => $bug_id);
	my ($bug) = $req->next('bug');
	my ($bug_old) = $req->next('bug_previous');
	my ($bug_log) = $req->next('bug_log');

	push(@cc, $bug_old->{owner});
	push(@cc, $bug->{owner});

	$subject = "[bug $bug_id] $bug->{summary}";
	$email = <<EOMAIL;
Changes to bug $bug_id entered by $bug->{last_user}

http://www.peppler.org/bugdb/view?bug_id=$bug_id

EOMAIL

    $email .= sprintf("%15s %20s %20s\n\n", 'Category', 'From', 'To');
        foreach my $key (qw(platform product os owner component severity
			    status priority resolution)) 
	{
	    if($bug->{$key} ne $bug_old->{$key}) {
		$email .= sprintf("%15s %20s %20s\n", 
				  $key, tab_lookup($key, $bug_old->{$key}),
				  tab_lookup($key, $bug->{$key}));
	    }
	}
	if($bug_log->{log_date} eq $bug->{lastmod_date}) {
	    $email .= "
Added comment by $bug_log->{email}:

$bug_log->{log_text}\n";
	}
    } elsif($type eq 'logentry') {
    }

    $req->listCc(bug_id => $bug_id);
    while(my $d = $req->next('cc_list')) {
	push(@cc, $d->{email});
    }
    
    $recipients = join(',', @cc);

    warn "sendmail: $email\n";
    warn "sendmail: @cc\n";

    open(MAIL, "|/usr/sbin/sendmail -t") or die "Can't open sendmail: $!";
    print MAIL "From: bugdb\@peppler.org\n";
    print MAIL "To: $recipients\n";
    print MAIL "Subject: $subject\n\n";
    print MAIL $email;
    close(MAIL);
}

sub tab_lookup {
    my $key   = shift;
    my $value = shift;

    my $list;
    if($key eq 'severity') {
	$list = $severity_list;
    } elsif($key eq 'priority') {
	$list = $priority_list;
    } elsif($key eq 'status') {
	$list = $status_list;
    } elsif($key eq 'resolution') {
	$list = $resolution_list;
    }

    return $value unless $list;

    foreach (@$list) {
	if($_->{id} == $value) {
	    return $_->{name};
	}
    }

    return $value;
}


1;
